//
// Test program for RJ45 connected, 6 pixel per unit, 'art board' project.
// Everything is build around '3'..
// 3 LEDs per pixel.  3 pixels per half board. 
//

/*
--NOTES ON RAM USAGE:--

Neopixel raw data:
186 pixels = 30x 6 LED modules. + 2x 3led debug modules, 3 bytes per pixel: 558 bytes RAM.

Module control and temp data:
24 modules + 2 debug modules: 4 bytes per module for config + 6 bytes for temp data:  250 bytes RAM


*/

#include <Adafruit_NeoPixel.h>


// This needs to be count + 2  (3 pixels at each end)
#define MODULE_COUNT 24+2
// Whatever the above number, this one is the last /actual/ in the string.
// 0 is first debug.
#define LASTMODULE 4


// How many pixels are attached
#define PIXEL_COUNT 186  // default. 

// From: simple_new_operator.ino
// Rather than declaring the whole NeoPixel object here, we just create
// a pointer for one, which we'll then allocate later...
//Adafruit_NeoPixel *pixels;


// Which pin on the Arduino is connected to the pixel string?
#define PIXEL_PIN    6

#define PIXEL_FORMAT NEO_GRB + NEO_KHZ800
//   NEO_GRB     Pixels are wired for GRB bitstream (most NeoPixel products)
//   NEO_RGB     Pixels are wired for RGB bitstream (v1 FLORA pixels, not v2)

Adafruit_NeoPixel pixels = Adafruit_NeoPixel(PIXEL_COUNT, PIXEL_PIN, PIXEL_FORMAT);


//(Lookup table for how many and what type of displays.
// Static data: id(0 is nothing), Num:(how many 'sets of 3'), Type:(Round(1), long(2), custom(3)), Mode:(See below), Speed(*10 mS)
// Mode: 0 = static, 1 = fade up/down, 2 = fade between in and out, 3 = rainbow spin, 4 = rainbow fade up/down, 5 = Twinkle, 6 = Sparkle
// 7 = Fire,  254 = Debug (Simple count)
// {leds,Type,Mode,Delay}
#define LEDT_LEDS 0
#define LEDT_TYPE 1
#define LEDT_MODE 2
#define LEDT_DELAY 3
byte ledTable[MODULE_COUNT][4] = {
  {1,3,254,25},  // 'Module 0':  (This is used for debug/start of string LEDs if they exist)
  {2,1,1,1},
  {2,1,1,10},
  {2,1,0,0},
  {2,1,0,0},
  {1,1,0,0},
  {2,1,0,0},    
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {1,1,0,0},
  {2,1,0,0},    
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},
  {2,1,0,0},  
  {2,1,0,0},
  {1,3,254,0}   // Also nothing. End of string.
};

// Dynamic data: r, g, b, timeval, tempval1, tempval2
#define LEDD_RED 0
#define LEDD_GREEN 1
#define LEDD_BLUE 2
#define LEDD_TIME 3
#define LEDD_TMP1 4
#define LEDD_TMP2 5
byte ledData[MODULE_COUNT][6] = { };

unsigned long updateMillis = 0;        // Last time update was run
unsigned long loopMillis = 0;        // Save how long updates took

// setup() function -- runs once at startup --------------------------------

void setup() {
  int numpixels=0;

  // Open serial port
  Serial.begin(115200);
  Serial.println("\n\n");
  Serial.println("RJ45 Art Board Driver.");

  Serial.println("Init:  ");  
  randomSeed(analogRead(0));

  Serial.print("  Data:");    
  for(int modu=0; modu<MODULE_COUNT; modu++) { // For each module
    numpixels=numpixels+(ledTable[modu][LEDT_LEDS]*3);  // Work out how many pixels we have in total
    ledData[modu][LEDD_RED]=0;  
    ledData[modu][LEDD_GREEN]=0;  // And make sure data table is blank
    ledData[modu][LEDD_BLUE]=0;
    ledData[modu][LEDD_TIME]=0;
    ledData[modu][LEDD_TMP1]=0;
    ledData[modu][LEDD_TMP2]=0;
  }
  Serial.println(".");    

  // Then create a new NeoPixel object dynamically with these values:
//  pixels = new Adafruit_NeoPixel(numpixels, LED_PIN, PIXEL_FORMAT);
  pixels.updateLength(numpixels);
  
  Serial.print("  Modules:");
  Serial.print(MODULE_COUNT);
  Serial.print(" Pixels:");
  Serial.println(pixels.numPixels());


  Serial.print("  Pixels:");  
  pixels.begin();           // INITIALIZE NeoPixel strip object (REQUIRED)
  pixels.show();            // Turn OFF all pixels ASAP
  pixels.setBrightness(20); // Set BRIGHTNESS to something low (max = 255)
  
  Serial.print(" Red");  
  // Quick once off test to make sure everything works
  colorWipe(pixels.Color(255,   0,   0), 10); // Red
  delay(100);
  Serial.print(". Green");    
  colorWipe(pixels.Color(  0, 255,   0), 10); // Green
  delay(100);
  Serial.print(". Blue");    
  colorWipe(pixels.Color(  0,   0, 255), 10); // Blue
  delay(100);
  Serial.print(". Off");    
  colorWipe(pixels.Color(  0,   0,   0), 50); // Off
  Serial.println(". done");

  //pixels.setBrightness(255); // Set BRIGHTNESS to full for main program.
  pixels.setBrightness(32); // Set BRIGHTNESS to About a quarter for testing
  #define DEBUG_BRIGHT 100

  Serial.println("  Random:");
  //  Set up some random data and modules each time we start. So every time the resulting display is different.
  // Ignore first and last modules
  for(int modu=1; modu<MODULE_COUNT-1; modu++) { // For each module
    module_rgb(modu,random(0,255),random(0,255),random(0,255));  // Some colours
    ledTable[modu][LEDT_MODE]=random(0,6); // Some modes
   // ledTable[modu][LEDT_MODE]=1;
    ledTable[modu][LEDT_DELAY]=random(5,20); // Some delays
    
    Serial.print("    M:");
    Serial.print(modu);
    Serial.print(" RGB(");
    Serial.print(ledData[modu][LEDD_RED],HEX);
    Serial.print(",");
    Serial.print(ledData[modu][LEDD_GREEN],HEX);
    Serial.print(",");
    Serial.print(ledData[modu][LEDD_BLUE],HEX);
    Serial.print(") Mode:");
    Serial.print(ledTable[modu][LEDT_MODE]);    
    Serial.print(" Delay:");
    Serial.println(ledTable[modu][LEDT_DELAY]);    
  }

  // Pre-load any values we need for the display
  // After random, to force status
//  module_rgb(1,255,0,0);    // Module 1 is red
//  module_rgb(2,128,0,128);  // Module 2 is yellow
  //module_rgb(2,10,0,10);  // Module 2 is yellow
  
//  module_rgb(1,255,0,0);    // Module 1 is red
//  module_rgb(2,128,0,128);  // Module 2 is yellow
  //module_rgb(2,10,0,10);  // Module 2 is yellow
  
 // ledData[2][LEDD_TMP1]=250;  // Start Module 2's cycle at full brightness
 // ledData[2][LEDD_TMP2]=1;  

    // Last actual module in string
    ledTable[LASTMODULE][LEDT_LEDS]=1;    // 3 LEDs
    ledTable[LASTMODULE][LEDT_MODE]=254;   // Debug mode
    ledTable[LASTMODULE][LEDT_DELAY]=ledTable[0][3]*3;    // Delay (Twice first 3 pixel debug)

  
  Serial.println("Done.\n");  
  delay(500);


}




// loop() function - Round and round it goes ---------------

void loop() {
unsigned long currentMillis = millis();  // Make sure every loop is the same 'time'

/*
  module_rgb(1,255,0,0);
  module_rgb(2,255,0,255);
  module_update(0);
  delay(200);  
  
  module_rgb(1,0,255,0);
  module_rgb(2,0,255,255);
  module_update(0);
  delay(200);  

  module_rgb(1,0,0,255);
  module_rgb(2,255,255,0);
*/

  if(currentMillis-updateMillis>10){    // Every 10mS
    Serial.print("<");
    Serial.print(loopMillis-updateMillis);   // How long module_update(0) took last time.
    Serial.print(">");
    updateMillis=currentMillis;   // Save for next time.
    module_update(0);  // Update all the modules!
    loopMillis=millis();  // Profiling loop.

}
//  delay(10); // Small delay just to make things human scale
  
//  Serial.println("");
}

// Set up data for module
void module_rgb(int modnum, byte r, byte g, byte b) {
  ledData[modnum][LEDD_RED]=r;
  ledData[modnum][LEDD_GREEN]=g;
  ledData[modnum][LEDD_BLUE]=b;  
}


// Update modules. Any modnum over zero is update only that one
// modnum = 0:  Update ALL modules
void module_update(int modnum) {
  Serial.print("Updating:");
  
  for(int modu=0; modu<MODULE_COUNT; modu++) { // For each module
    if(modnum>0) { modu = modnum; }  // Force current module to selected one
    Serial.print(modu);
    
    switch(ledTable[modu][LEDT_MODE]) {  // Need to know what we are doing to each module
      case 0: // static
        modu_static(modu); 
      break;
      case 1: // fade up/down,
        modu_fadeupdn(modu); 
      break;
      case 2: // fade between in and out
        modu_fadeinout(modu);       
      break;
      case 3: // rainbow spin
        modu_rainbowspin(modu);
      break;
      case 4: // rainbow fade up/down
        modu_rainbowfade(modu);
      break;
      case 5: // Twinkle
        modu_twinkle(modu);
      break;
      case 6: // Sparkle
        modu_sparkle(modu);
      break;
      case 7: // Fire
 //       modu_fire(modu);
      break;      
      case 254: // Debug
        modu_debug(modu);
      break;
    }
    
    if(modnum>0) { modu = MODULE_COUNT;  }  // Get out of the loop by skipping to the end.

    Serial.print(",");
  }
  
  pixels.show();                          //  Update physical strip to match updated pixels
  Serial.println(" .done");

}

// What is in ledData gets written out to the pixel string in the right location
void modu_static(int modnum) {
  Serial.print("(");
  
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) {               // Go through every module before this.
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);   // Count number of pixels
  }

  byte r = ledData[modnum][LEDD_RED];
  byte g = ledData[modnum][LEDD_GREEN];
  byte b = ledData[modnum][LEDD_BLUE];
  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module
    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
  Serial.print(".)");
  
}

// What is in ledData gets fades the whole module down to zero and back up
// LEDD_TMP1 stores current brightness, LEDD_TMP2 is direction
void modu_fadeupdn(int modnum) {
  Serial.print("(");
    
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) { 
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }
  
  byte r,g,b;

  //// We want to make things fade roughly the same despite the brightness of inputs.
  //byte rgbmax = max (max (ledData[modnum][LEDD_RED], ledData[modnum][LEDD_GREEN]), ledData[modnum][LEDD_GREEN]);   // Work out the highest value we will use
  //byte scalevalue=rgbmax/25;  // How many steps the fade should be by
  //if(scalevalue==0) { scalevalue =1;  }  // Make sure we can't go lower.
  //Serial.print(scalevalue);
  //Serial.print(":");
  byte scalevalue=10;
  

  int fade=ledData[modnum][LEDD_TMP1];  // Use int to make sure we can go negative and posative
  if(ledData[modnum][LEDD_TMP2]==1){  // Brightness going up
    fade=fade+scalevalue;
    if(fade>=255) {
      fade=255;  // Make sure it's not bigger than byte
      ledData[modnum][LEDD_TMP2]=0;  // Swap directions
    }
  } else { // Going down
    fade=fade-scalevalue;
    if(fade<=0) {
      fade=0;  // Make sure it's not negative
      ledData[modnum][LEDD_TMP2]=1;  // Swap directions
    }    
  }
  ledData[modnum][LEDD_TMP1]=(byte) fade;  // Save the value again
  Serial.print(ledData[modnum][LEDD_TMP1]);

  // From https://github.com/adafruit/Adafruit_NeoPixel/blob/master/Adafruit_NeoPixel.cpp  ( void Adafruit_NeoPixel::setPixelColor(  )
  // Does the brightness caculations for the colours.
  r = pixels.gamma8((ledData[modnum][LEDD_RED] * ledData[modnum][LEDD_TMP1]) >> 8);
  g = pixels.gamma8((ledData[modnum][LEDD_GREEN] * ledData[modnum][LEDD_TMP1]) >> 8);
  b = pixels.gamma8((ledData[modnum][LEDD_BLUE] * ledData[modnum][LEDD_TMP1]) >> 8);
      
  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module
    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
  Serial.print(")"); 
}


// What is in ledData gets fades between in and outer rings.   (Not sure what to do with 3 pixel modules)
// LEDD_TMP1 stores current brightness, LEDD_TMP2 is direction
void modu_fadeinout(int modnum) {
  Serial.print("(");
    
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) {
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }

  byte r,g,b;
  
  if(ledData[modnum][LEDD_TMP2]==1){  // Brightness going up
    ledData[modnum][LEDD_TMP1]=ledData[modnum][LEDD_TMP1]+5;
    if(ledData[modnum][LEDD_TMP1]==255) {
      ledData[modnum][LEDD_TMP2]=0;  // Swap directions
    }
  } else { // Going down
    ledData[modnum][LEDD_TMP1]=ledData[modnum][LEDD_TMP1]-5;
    if(ledData[modnum][LEDD_TMP1]==0) {
      ledData[modnum][LEDD_TMP2]=1;  // Swap directions
    }    
  }
  Serial.print(ledData[modnum][LEDD_TMP1]);

  // From https://github.com/adafruit/Adafruit_NeoPixel/blob/master/Adafruit_NeoPixel.cpp  ( void Adafruit_NeoPixel::setPixelColor(  )
  // Does the brightness caculations for the colours.
  r = pixels.gamma8((ledData[modnum][LEDD_RED] * ledData[modnum][LEDD_TMP1]) >> 8);
  g = pixels.gamma8((ledData[modnum][LEDD_GREEN] * ledData[modnum][LEDD_TMP1]) >> 8);
  b = pixels.gamma8((ledData[modnum][LEDD_BLUE] * ledData[modnum][LEDD_TMP1]) >> 8);

  // First 3 pixels
  if(ledTable[modnum][LEDT_LEDS]>=1) {   
    for(int led=0; led<3; led++) { // For each led in module
      pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
    }
  }

  // If they exist, the next three pixels. Ignore any others.
  // Maybe try using  "(ledTable[modnum][LEDT_LEDS]*3)-3)" to fill every LED after first 3.
  if(ledTable[modnum][LEDT_LEDS]>=2) {
    r = pixels.gamma8((ledData[modnum][LEDD_RED] * (255-ledData[modnum][LEDD_TMP1])) >> 8);
    g = pixels.gamma8((ledData[modnum][LEDD_GREEN] * (255-ledData[modnum][LEDD_TMP1])) >> 8);
    b = pixels.gamma8((ledData[modnum][LEDD_BLUE] * (255-ledData[modnum][LEDD_TMP1])) >> 8);
    pixoffset=pixoffset+3;
    for(int led=0; led<3; led++) { // For each led in module
      pixels.setPixelColor(pixoffset+led, r, g, b);  //  Inverted brightness
    }
  }
  
  Serial.print(")"); 
}


// Rainbow spin on 3 or 6 LEDs.
// LEDD_TMP1 stores current part of cycle
void modu_rainbowspin(int modnum) {
  Serial.print("(");
    
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) {
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }

  byte r,g,b;
  
  int wheelval = ledData[modnum][LEDD_TMP1];   // Make to an int so we can overflow 255
  wheelval=wheelval+(ledTable[modnum][LEDT_LEDS]*3);  // Count up to 255, in number of pixels the module has
  if(wheelval>255) {
    wheelval=0;  // Back to zero
  }
  ledData[modnum][LEDD_TMP1]=wheelval;
  
  Serial.print(ledData[modnum][LEDD_TMP1]);
  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module

    uint32_t packedcolor = Wheel(ledData[modnum][LEDD_TMP1]+(led*10));
    packedcolor=pixels.gamma32(packedcolor); // 
    r = Red(packedcolor);
    g = Green(packedcolor);
    b = Blue(packedcolor);
    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
  
  Serial.print(")"); 
}



// What fades the whole module down to zero and back up with diffrent rainbow each time.
// LEDD_TMP1 stores current brightness, // LEDD_TMP1, use bit 8 for direction. 
// LEDD_TMP2 is rainbow wheel location
void modu_rainbowfade(int modnum) {
  Serial.print("(");
    
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) { 
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }
  
  byte r,g,b;

  byte fade=ledData[modnum][LEDD_TMP2]&0b01111111;
  byte dir=(ledData[modnum][LEDD_TMP2]&0b10000000) >> 7 ;
  
  if(dir==1){  // Brightness going up
    fade=fade+5;
    if(fade==125) {
      dir=0;  // Swap directions
    }
  } else { // Going down
    fade=fade-5;
    if(fade==0) {
      dir=1;  // Swap directions
    }    
  }
  ledData[modnum][LEDD_TMP2]=fade|(dir<<7);


  int wheelval = ledData[modnum][LEDD_TMP1];   // Make to an int so we can overflow 255

  // Move wheel on each fade
  if(fade==0) {
    wheelval=wheelval+(ledTable[modnum][LEDT_LEDS]*3);  // Count up to 255, in number of pixels the module has
    if(wheelval>255) {
      wheelval=0;  // Back to zero
    }
    ledData[modnum][LEDD_TMP1]=(byte) wheelval;  // Save once we are done
  }

    Serial.print(fade);
    Serial.print(".");
    Serial.print(wheelval);

  fade=fade*2;  // Full range again.
  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module

    uint32_t packedcolor = Wheel(wheelval+(led*5));
    packedcolor=pixels.gamma32(packedcolor);
    
    // Does the brightness caculations for the colours.
    r = (Red(packedcolor) * fade) >> 8;
    g = (Green(packedcolor) * fade) >> 8;
    b = (Blue(packedcolor) * fade) >> 8;

    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
      
  /*
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module
    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
  */
  Serial.print(")"); 
}



// Randomly light LEDs one of the LEDs on a module.  Saves prev pos in LEDD_TMP1 to prevent visual pauses.
// Based on: https://www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/#LEDStripEffectRandomColorTwinkle
void modu_twinkle(int modnum) {
  Serial.print("(");
  
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) { 
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }

  //  All pixels to zero
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) {
    pixels.setPixelColor(pixoffset+led, 0, 0, 0);         
  }

  byte whatpix=ledData[modnum][LEDD_TMP1];
  // Make sure we pick a diffrent one each time (For visual effect)
  do {
    whatpix = random(0,ledTable[modnum][LEDT_LEDS]*3);
  } while(whatpix==ledData[modnum][LEDD_TMP1]);   // Keep going until we get a diffrent one!

  // Set our single pixel a random set of colours!
  if(ledData[modnum][LEDD_RED]&ledData[modnum][LEDD_GREEN]&ledData[modnum][LEDD_BLUE]) {
    pixels.setPixelColor(pixoffset+whatpix, random(0,255),random(0,255),random(0,255));  
  } else {
    pixels.setPixelColor(pixoffset+whatpix, ledData[modnum][LEDD_RED],ledData[modnum][LEDD_GREEN],ledData[modnum][LEDD_BLUE]);  
  }

  ledData[modnum][LEDD_TMP1]=whatpix;   // Save what pixel we set for next time
  Serial.print(whatpix,DEC);
  Serial.print(")");
  
}


// Randomly light LEDs one of the LEDs on a module for a single cycle only. Uses colour data in ledData
// Saves prev pos in LEDD_TMP1 to prevent visual pauses.  LEDD_TMP2 saves if we need to blank it next cycle.
// Based on: https://www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/#LEDStripEffectSparkle
void modu_sparkle(int modnum) {
  Serial.print("(");

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) { 
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }

  // If this is set, it means we need to turn it off on the next refresh cycle rather than wait for the delay
  if(ledData[modnum][LEDD_TMP2]>0) {
    //  All pixels to zero
    for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) {
      pixels.setPixelColor(pixoffset+led, 0, 0, 0);         
    }
    ledData[modnum][LEDD_TMP2]=0;  // Make sure we only do it once.
    Serial.print("!");
  }
  
  // Real Delay before change value
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }
  

  //  All pixels to zero  (We allways want them off)
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) {
    pixels.setPixelColor(pixoffset+led, 0, 0, 0);         
  }

  byte whatpix=ledData[modnum][LEDD_TMP1];
  // Make sure we pick a diffrent one each time (For visual effect)
  // Should be less important for 'sparkle' but it dosen't really hurt.
  do {
    whatpix = random(0,ledTable[modnum][LEDT_LEDS]*3);
  } while(whatpix==ledData[modnum][LEDD_TMP1]);

  byte r = ledData[modnum][LEDD_RED];
  byte g = ledData[modnum][LEDD_GREEN];
  byte b = ledData[modnum][LEDD_BLUE];
  
  // Set our single pixel a random set of colours!
  pixels.setPixelColor(pixoffset+whatpix, r,g,b);
  ledData[modnum][LEDD_TMP2]=1; // Turn it off next cycle

  ledData[modnum][LEDD_TMP1]=whatpix;
  Serial.print(whatpix,DEC);
  Serial.print(")");
  
}



// Fire effect.  Fade overall module up and down using reds and oranges. With flickers of white for sparks.
// Ideas:  Up and down brightness is simple cycle controlled by random selection. Maybe sawtooth?
// Sparks are based on 'flicker' code.
//
// LEDD_TMP1 stores current brightness, LEDD_TMP1, use bit 8 for direction. LEDD_TMP2 is rainbow wheel location
void modu_fire(int modnum) {
  Serial.print("(");
    
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) { 
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);
  }
  
  byte r,g,b;

  byte fade=ledData[modnum][LEDD_TMP2]&0b01111111;
  byte dir=(ledData[modnum][LEDD_TMP2]&0b10000000) >> 7 ;
  
  if(dir==1){  // Brightness going up
    fade=fade+5;
    if(fade==125) {
      dir=0;  // Swap directions
    }
  } else { // Going down
    fade=fade-5;
    if(fade==0) {
      dir=1;  // Swap directions
    }    
  }
  ledData[modnum][LEDD_TMP2]=fade|(dir<<7);


  int wheelval = ledData[modnum][LEDD_TMP1];   // Make to an int so we can overflow 255

  // Move wheel on each fade
  if(fade==0) {
    wheelval=wheelval+(ledTable[modnum][LEDT_LEDS]*3);  // Count up to 255, in number of pixels the module has
    if(wheelval>255) {
      wheelval=0;  // Back to zero
    }
    ledData[modnum][LEDD_TMP1]=(byte) wheelval;  // Save once we are done
  }

    Serial.print(fade);
    Serial.print(".");
    Serial.print(wheelval);

  fade=fade*2;  // Full range again.
  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module

    uint32_t packedcolor = Wheel(wheelval+(led*5));
    packedcolor=pixels.gamma32(packedcolor);
    
    // Does the brightness caculations for the colours.
    r = (Red(packedcolor) * fade) >> 8;
    g = (Green(packedcolor) * fade) >> 8;
    b = (Blue(packedcolor) * fade) >> 8;

    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
      
  /*
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module
    pixels.setPixelColor(pixoffset+led, r, g, b);         //  Set pixel's color (in RAM)
  }
  */
  Serial.print(")"); 
}



// Debug module. Make sure data is moving and complete.
// A nice simple one. Just display binary and update each delay cycle. Stored in LEDD_TMP1
void modu_debug(int modnum) {
  Serial.print("(");
  
  // Delay before change value.
  if(ledData[modnum][LEDD_TIME]>0) {   // While there is still delay in use
    ledData[modnum][LEDD_TIME]--;      // Keep counting down
    Serial.print(")");
    return(0);                 // And drop out of function here.
  } else {
    ledData[modnum][LEDD_TIME]=ledTable[modnum][LEDT_DELAY];  // Reset delay counter from preset value
  }

  // Work out offset to this module
  int pixoffset=0;
  for(int i=0; i<modnum; i++) {               // Go through every module before this.
    pixoffset=pixoffset+(ledTable[i][LEDT_LEDS]*3);   // Count number of pixels
  }

 // Count from zero to 15(3)
 ledData[modnum][LEDD_TMP1]++;
 if(ledData[modnum][LEDD_TMP1]>2){ 
    ledData[modnum][LEDD_TMP1]=0;
 }
 Serial.print(ledData[modnum][LEDD_TMP1]);

  
  for(int led=0; led<(ledTable[modnum][LEDT_LEDS]*3); led++) { // For each led in module
    
//    if((ledData[modnum][LEDD_TMP1]>>led)&0b00000001) {
    if(ledData[modnum][LEDD_TMP1]==led) {

      pixels.setPixelColor(pixoffset+led, DEBUG_BRIGHT, 0, 0);         //  Red
    } else {
      pixels.setPixelColor(pixoffset+led, 0, DEBUG_BRIGHT, 0);         //  Green
      
    }
  }
  Serial.print(")");
  
}


// Whole strip stuff, not module stuff


void striptests() {

  
  // Fill along the length of the strip in various colors...
  Serial.print("R");
  x3_colourWipe(pixels.Color(255,  0,  0), 100,1); // Red
  Serial.print("G");
  x3_colourWipe(pixels.Color(  0,255,  0), 100,1); // Green
  Serial.print("B");
  x3_colourWipe(pixels.Color(  0,  0,255), 100,1); // Blue
  Serial.print("P");
  x3_colourWipe(pixels.Color(255,  0,255), 100,2); // Purple
  Serial.print("C");
  x3_colourWipe(pixels.Color(  0,255,255), 100,2); // Cyan/Aqua
  Serial.print("Y");
  x3_colourWipe(pixels.Color(255,255,  0), 100,3); // Yellow

  rainbow(10);             // Flowing rainbow cycle along the whole strip

}
  
void x3_colourWipe(uint32_t color, int wait, int skip) {
  
  for(int modu=0; modu<MODULE_COUNT; modu=modu+skip) { // For each module
    for(int led=0; led<(ledTable[modu][LEDT_LEDS]*3); led++) { // For each led
      pixels.setPixelColor((modu*6)+led, color);         //  Set pixel's color (in RAM)
      pixels.show();                          //  Update strip to match
      delay(wait);                           //  Pause for a moment
    }
     
    delay(wait*2);                           //  Pause for a longer moment
  }
  
}


void colourAll(uint32_t color, int wait, int skip) {
  
  for(int modu=0; modu<MODULE_COUNT; modu=modu+skip) { // For each module
    for(int led=0; led<ledTable[modu][LEDT_LEDS]; led++) { // For each led
      pixels.setPixelColor((modu*6)+led, color);         //  Set pixel's color (in RAM)
    }
     
    delay(wait*2);                           //  Pause for a longer moment
  }
  
}


// Some functions of our own for creating animated effects -----------------

// Fill strip pixels one after another with a color. Strip is NOT cleared
// first; anything there will be covered pixel by pixel. Pass in color
// (as a single 'packed' 32-bit value, which you can get by calling
// pixels.Color(red, green, blue) as shown in the loop() function above),
// and a delay time (in milliseconds) between pixels.
void colorWipe(uint32_t color, int wait) {
  for(int i=0; i<pixels.numPixels(); i++) { // For each pixel in pixels...
    pixels.setPixelColor(i, color);         //  Set pixel's color (in RAM)
    pixels.show();                          //  Update strip to match
    delay(wait);                           //  Pause for a moment
  }
}

// Theater-marquee-style chasing lights. Pass in a color (32-bit value,
// a la pixels.Color(r,g,b) as mentioned above), and a delay time (in ms)
// between frames.
void theaterChase(uint32_t color, int wait) {
  for(int a=0; a<10; a++) {  // Repeat 10 times...
    for(int b=0; b<3; b++) { //  'b' counts from 0 to 2...
      pixels.clear();         //   Set all pixels in RAM to 0 (off)
      // 'c' counts up from 'b' to end of strip in steps of 3...
      for(int c=b; c<pixels.numPixels(); c += 3) {
        pixels.setPixelColor(c, color); // Set pixel 'c' to value 'color'
      }
      pixels.show(); // Update strip with new contents
      delay(wait);  // Pause for a moment
    }
  }
}

// Rainbow cycle along whole pixels. Pass delay time (in ms) between frames.
void rainbow(int wait) {
  // Hue of first pixel runs 5 complete loops through the color wheel.
  // Color wheel has a range of 65536 but it's OK if we roll over, so
  // just count from 0 to 5*65536. Adding 256 to firstPixelHue each time
  // means we'll make 5*65536/256 = 1280 passes through this outer loop:
  for(long firstPixelHue = 0; firstPixelHue < 5*65536; firstPixelHue += 256) {
    for(int i=0; i<pixels.numPixels(); i++) { // For each pixel in pixels...
      // Offset pixel hue by an amount to make one full revolution of the
      // color wheel (range of 65536) along the length of the strip
      // (pixels.numPixels() steps):
      int pixelHue = firstPixelHue + (i * 65536L / pixels.numPixels());
      // pixels.ColorHSV() can take 1 or 3 arguments: a hue (0 to 65535) or
      // optionally add saturation and value (brightness) (each 0 to 255).
      // Here we're using just the single-argument hue variant. The result
      // is passed through pixels.gamma32() to provide 'truer' colors
      // before assigning to each pixel:
      pixels.setPixelColor(i, pixels.gamma32(pixels.ColorHSV(pixelHue)));
    }
    pixels.show(); // Update strip with new contents
    delay(wait);  // Pause for a moment
  }
}

// Rainbow-enhanced theater marquee. Pass delay time (in ms) between frames.
void theaterChaseRainbow(int wait) {
  int firstPixelHue = 0;     // First pixel starts at red (hue 0)
  for(int a=0; a<30; a++) {  // Repeat 30 times...
    for(int b=0; b<3; b++) { //  'b' counts from 0 to 2...
      pixels.clear();         //   Set all pixels in RAM to 0 (off)
      // 'c' counts up from 'b' to end of strip in increments of 3...
      for(int c=b; c<pixels.numPixels(); c += 3) {
        // hue of pixel 'c' is offset by an amount to make one full
        // revolution of the color wheel (range 65536) along the length
        // of the strip (pixels.numPixels() steps):
        int      hue   = firstPixelHue + c * 65536L / pixels.numPixels();
        uint32_t color = pixels.gamma32(pixels.ColorHSV(hue)); // hue -> RGB
        pixels.setPixelColor(c, color); // Set pixel 'c' to value 'color'
      }
      pixels.show();                // Update strip with new contents
      delay(wait);                 // Pause for a moment
      firstPixelHue += 65536 / 90; // One cycle of color wheel over 90 frames
    }
  }
}


// Input a value 0 to 255 to get a color value.
// The colours are a transition r - g - b - back to r.
uint32_t Wheel(byte WheelPos) {
  WheelPos = 255 - WheelPos;
  if(WheelPos < 85) {
    return pixels.Color(255 - WheelPos * 3, 0, WheelPos * 3);
  }
  if(WheelPos < 170) {
    WheelPos -= 85;
    return pixels.Color(0, WheelPos * 3, 255 - WheelPos * 3);
  }
  WheelPos -= 170;
  return pixels.Color(WheelPos * 3, 255 - WheelPos * 3, 0);
}

// Returns the Red component of a 32-bit color
uint8_t Red(uint32_t color)
{
    return (color >> 16) & 0xFF;
}
 
// Returns the Green component of a 32-bit color
uint8_t Green(uint32_t color)
{
    return (color >> 8) & 0xFF;
}
 
// Returns the Blue component of a 32-bit color
uint8_t Blue(uint32_t color)
{
    return color & 0xFF;
}

    
